<?php
/**
 * Description of DHL_Paket_DE_Versenden_REST_API
 *
 * @author mario
 */
class DHL_basic
{
    public function __construct( array $data ){
        $props = $this::$propertyTypes;
        foreach($props as $key => $p){
            if(isset($p['mandatory']) and $p['mandatory'] == true){
                if(!isset($data[ $key ] )){
                    throw new Exception(sprintf("Class %s - Property %s is mandatory - no value found",
                        get_class($this), $key, $type,$props[$key]['type']
                    ));
                }
            }
            if($type == 'string' and !empty($data[$key])){
                if(strlen($data[$key]) < $p['min']){
                    throw new Exception(sprintf("Class %s - Property %s has min length %d - the value '%s' has only %d",
                        get_class($this), $key, $p['min'],$data[$key], strlen($data[$key])
                    ));
                }
                if(isset($p['max']) and $p['max'] > 0){
                    $data[$key] = substr($data[$key],0,$p['max']);
                }
            }
        }
        foreach($data as $k=>$v){
            if(!isset($props[$k])){
                throw new Exception(sprintf("Class %s __construct() with unknown Class Var '%s'",
                        get_class($this), $k
                ));
            }
            $type = gettype($v);
            $chk = false;
            switch($type){
                case 'integer':
                case 'double':
                case 'string':
                case 'boolean':
                case 'array':
                    break;
                case 'NULL':
                    break;
                default:
                    if(get_class($v) != $props[$k]['type']){
                        throw new Exception(sprintf("Class %s - Property %s has Value with type %s, but expecting %s",
                                get_class($this), $k, get_class($v),$props[$k]['type']
                        ));
                    }
                    $chk = true;
                    break;
            }
            if(!$chk and $type != $props[$k]['type']){
                throw new Exception(sprintf("Class %s - Property %s has Value with type %s, but expecting %s",
                    get_class($this), $k, $type,$props[$k]['type']
                ));
            }
            $this->$k = $v;
        }
    }
    
    public static function get_input_fields(){
        $props = $this::$propertyTypes;
        foreach($props as $key => $p){
        }
        return 'Hallo';
    }
}

class DHL_Services extends DHL_basic
{
    public static $propertyTypes = [
        'preferredNeighbour' => [
            'type' => 'string',
            'min'=>0,
            'max'=>100,
        ],
        'preferredLocation' => [
            'type' => 'string',
            'min'=>0,
            'max'=>100,
        ],
        'visualCheckOfAge' => [
            'type' => 'string',
            'min'=>3,
            'max'=>3,
        ],
        'namedPersonOnly' => [
            'type' => 'boolean',
        ],
        'identCheck' => [
            'type' => 'DHL_Service_identCheck',
        ],
        'signedForByRecipient' => [
            'type' => 'boolean',
        ],
        'endorsement' => [
            'type' => 'string',
        ],
        'preferredDay' => [
            'type' => 'string',
        ],
        'noNeighbourDelivery' => [
            'type' => 'boolean',
        ],
        'additionalInsurance' => [
            'type' => 'DHL_Service_Amount',
        ],
        'bulkyGoods' => [
            'type' => 'boolean',
        ],
        'cashOnDelivery' => [
            'type' => 'DHL_Service_cashOnDelivery',
        ],
        'premium' => [
            'type' => 'boolean',
        ],
        'closestDropPoint' => [
            'type' => 'boolean',
        ],
        'parcelOutletRouting' => [
            'type' => 'string',
        ],
        'dhlRetoure' => [
            'type' => 'DHL_Service_dhlRetoure',
        ],
        'postalDeliveryDutyPaid' => [
            'type' => 'boolean',
        ]
    ];
}

class DHL_FullAdress  extends DHL_basic
{
    public static $propertyTypes = [
        'name1' => [
            'type' => 'string',
            'min'=>1,
            'max'=>50,
            'mandatory'=>true
        ],
        'name2' => [
            'type' => 'string',
            'min'=>1,
            'max'=>50,
        ],
        'name3' => [
            'type' => 'string',
            'min'=>1,
            'max'=>50,
        ],
        'addressStreet' => [
            'type' => 'string',
            'min'=>1,
            'max'=>50,
            'mandatory'=>true
        ],
        'additionalAddressInformation1' => [
            'type' => 'string',
            'min'=>1,
            'max'=>60,
        ],
        'additionalAddressInformation2' => [
            'type' => 'string',
            'min'=>1,
            'max'=>60,
        ],
        'postalCode' => [
            'type' => 'string',
            'min'=>3,
            'max'=>10,
        ],
        'city' => [
            'type' => 'string',
            'min'=>1,
            'max'=>40,
            'mandatory'=>true,
        ],
        'state' => [
            'type' => 'string',
            'min'=>1,
            'max'=>20,
        ],
        'country' => [
            'type' => 'string',
            'min'=>3,
            'max'=>3,
            'mandatory'=>true,
        ],
        'phone' => [
            'type' => 'string',
            'min'=>1,
            'max'=>20,
        ],
        'email' => [
            'type' => 'string',
            'min'=>3,
            'max'=>80,
        ]
    ];
}

class DHL_shipment extends DHL_basic
{
    public static $propertyTypes = [
        'product' => [
            'type' => 'string',
        ],
        'billingNumber'=>[
            'type'=>'string',
        ],
        'refNo'=>[
            'type'=>'string',
        ],
        'shipDate'=>[
            'type'=>'string',
        ],
        'shipper'=>[
            'type'=>'DHL_Shipper',
        ],
        'customs'=>[
            'type'=>'DHL_customs',
        ],
        'consignee'=>[
            'type'=>'DHL_FullAdress',
        ],
        'details'=>[
            'type'=>'DHL_Details',
        ],
        'services'=>[
            'type'=>'array'
        ]
    ];
}

// additionalInsurance, cashOnDelivery.amount
class DHL_Service_Amount extends DHL_basic
{
    public static $propertyTypes = [
        'currency' => [
            'type' => 'string',
        ],
        'value' => [
            'type' => 'double',
        ]
    ];
}

class DHL_Shipper extends DHL_basic
{
    public static $propertyTypes = [
        'name1' => [
            'type' => 'string',
            'min'=>1,
            'max'=>50,
            'mandatory'=>true
        ],
        'addressStreet' => [
            'type' => 'string',
            'min'=>1,
            'max'=>50,
            'mandatory'=>true
        ],
        'addressHouse' => [
            'type' => 'string',
            'min'=>1,
            'max'=>10,
        ],
        'postalCode' => [
            'type' => 'string',
            'min'=>3,
            'max'=>10,
        ],
        'city' => [
            'type' => 'string',
            'min'=>1,
            'max'=>40,
            'mandatory'=>true
        ],
        'country' => [
            'type' => 'string',
            'min'=>3,
            'max'=>3,
            'mandatory'=>true
        ],
        'shipperReference' => [
            'type' => 'DHL_shipperReference'
        ],
        'email' => [
            'type' => 'string',
            'min'=>3,
            'max'=>80,
        ]
    ];
}

class DHL_shipperReference extends DHL_basic
{
    public static $propertyTypes = [
        'shipperRef' => [
            'type' => 'string',
            'min'=>0,
            'max'=>50,
            'mandatory'=>true
        ]
    ];
}


class DHL_Service_identCheck extends DHL_basic
{
    public static $propertyTypes = [
        'firstName' => [
            'type' => 'string',
            'min'=>1,
            'max'=>35,
            'mandatory'=>true
        ],
        'lastName' => [
            'type' => 'string',
            'min'=>1,
            'max'=>35,
            'mandatory'=>true
        ],
        'dateOfBirth' => [
            'type' => 'string',
        ],
        'minimumAge' => [ // A16 or A18
            'type' => 'string',
            'min'=>3,
            'max'=>3,
        ]
    ];
}

class DHL_Service_cashOnDelivery extends DHL_basic
{
    public static $propertyTypes = [
        'amount' => [
            'type' => 'DHL_Service_Amount',
            'mandatory'=>true
        ],
        'bankAccount' => [
            'type' => 'DHL_Service_cashOnDelivery_bankAccount',
            'mandatory'=>true
        ],
        'transferNote1' => [
            'type' => 'String',
            'min'=>0,
            'max'=>35
        ],
        'transferNote2' => [
            'type' => 'String',
            'min'=>0,
            'max'=>35
        ]
    ];
}

class DHL_Service_cashOnDelivery_bankAccount extends DHL_basic
{
    public static $propertyTypes = [
        'accountHolder' => [
            'type' => 'string',
            'min'=>0,
            'max'=>80,
            'mandatory'=>true
        ],
        'bankName' => [
            'type' => 'string',
            'min'=>0,
            'max'=>80,
        ],
        'iban' => [
            'type' => 'string',
            'mandatory'=>true
        ],
        'bic'=> [
            'type'=>'string',
        ]
    ];
}

class DHL_Service_dhlRetoure extends DHL_basic
{
    public static $propertyTypes = [
        'billingNumber' => [
            'type' => 'string',
            'mandatory'=>true
        ],
        'refNo' => [
            'type' => 'string',
            'min'=>6,
            'max'=>50,
        ],
        'returnAddress' => [
            'type' => 'DHL_FullAdress',
        ]
    ];
}

class DHL_Weight  extends DHL_basic
{
    public static $propertyTypes = [
        'uom' => [
            'type' => 'string',
            'min'=>1,
            'max'=>2,
            'mandatory'=>true
        ],
        'value' => [
            'type' => 'integer',
            'mandatory'=>true
        ]
    ];
}

class DHL_Dim  extends DHL_basic
{
    public static $propertyTypes = [
        'uom' => [
            'type' => 'string',
            'min'=>2,
            'max'=>2,
            'mandatory'=>true
        ],
        'height' => [
            'type' => 'integer',
            'mandatory'=>true
        ],
        'length' => [
            'type' => 'integer',
            'mandatory'=>true
        ],
        'width' => [
            'type' => 'integer',
            'mandatory'=>true
        ]
    ];
}


class DHL_customs_item extends DHL_basic
{
    public static $propertyTypes = [
        'itemDescription' => [
            'type' => 'string',
            'min'=>1,
            'max'=>256,
            'mandatory'=>true,
        ],
        'countryOfOrigin' => [
            'type' => 'string',
            'min'=>3,
            'max'=>3
        ],
        'hsCode' => [
            'type' => 'string',
            'min'=>6,
            'max'=>11
        ],
        'packagedQuantity' => [
            'type' => 'integer',
            'mandatory'=>true,
        ],
        'itemValue' => [
            'type' => 'DHL_Service_Amount',
            'mandatory'=>true,
        ],
        'itemWeight' => [
            'type' => 'DHL_Weight',
            'mandatory'=>true,
        ]
    ];
}

class DHL_customs extends DHL_basic
{
    public static $propertyTypes = [
        'invoiceNo' => [
            'type' => 'string',
            'min'=>0,
            'max'=>35
        ],
        'exportType' => [
            'type' => 'string',
            'mandatory'=>true,
        ],
        'exportDescription' => [
            'type' => 'string',
            'min'=>0,
            'max'=>80
        ],
        'shippingConditions' => [
            'type' => 'string',
            'min'=>3,
            'max'=>3
        ],
        'permitNo' => [
            'type' => 'string',
            'min'=>0,
            'max'=>30
        ],
        'attestationNo' => [
            'type' => 'string',
            'min'=>0,
            'max'=>30
        ],
        'hasElectronicExportNotification' => [
            'type' => 'boolean',
        ],
        'MRN' => [
            'type' => 'string',
            'min'=>0,
            'max'=>18,
        ],
        'postalCharges' => [
            'type' => 'DHL_Service_Amount',
            'mandatory'=>true,
        ],
        'officeOfOrigin' => [
            'type' => 'string',
            'min'=>0,
            'max'=>35,
        ],
        'shipperCustomsRef' => [
            'type' => 'string',
            'min'=>0,
            'max'=>35,
        ],
        'consigneeCustomsRef' => [
            'type' => 'string',
            'min'=>0,
            'max'=>35,
        ],
        'items' => [
            'type' => 'array',
            'mandatory'=>true
        ]
    ];
}

class DHL_Locker  extends DHL_basic
{
    public static $propertyTypes = [
        'name' => [
            'type' => 'string',
            'min'=>1,
            'max'=>50,
            'mandatory'=>true
        ],
        'lockerID' => [
            'type' => 'integer',
            'mandatory'=>true
        ],
        'postNumber' => [
            'type' => 'string',
            'mandatory'=>true
        ],
        'city' => [
            'type' => 'string',
            'min'=>1,
            'max'=>40,
            'mandatory'=>true
        ],
        'country' => [
            'type' => 'string',
            'min'=>3,
            'max'=>3,
        ],
        'postalCode' => [
            'type' => 'string',
            'min'=>3,
            'max'=>10,
            'mandatory'=>true,
        ]
    ];
}

class DHL_PostOffice  extends DHL_basic
{
    public static $propertyTypes = [
        'name' => [
            'type' => 'string',
            'min'=>1,
            'max'=>50,
            'mandatory'=>true
        ],
        'retailID' => [
            'type' => 'integer',
            'mandatory'=>true
        ],
        'postNumber' => [
            'type' => 'string',
            'min' => 3,
            'max' => 10,
        ],
        'email' => [
            'type' => 'string',
            'min' => 3,
            'max' => 80,
        ],
        'city' => [
            'type' => 'string',
            'min' => 0,
            'max' => 80,
            'mandatory'=>true
        ],
        'country' => [
            'type' => 'string',
            'min' => 3,
            'max' => 3,
        ],
        'postalCode' => [
            'type' => 'string',
            'min' => 3,
            'max' => 10,
            'mandatory'=>true
        ]
    ];
}

class DHL_POBox  extends DHL_basic
{
    public static $propertyTypes = [
        'name1' => [
            'type' => 'string',
            'min'=>1,
            'max'=>50,
            'mandatory'=>true
        ],
        'name2' => [
            'type' => 'string',
            'min'=>1,
            'max'=>50,
        ],
        'name3' => [
            'type' => 'string',
            'min'=>1,
            'max'=>50,
        ],
        'poBoxID' => [
            'type' => 'integer',
            'mandatory'=>true
        ],
        'email' => [
            'type' => 'string',
            'min'=>3,
            'max'=>80,
        ],
        'city' => [
            'type' => 'string',
            'min'=>0,
            'max'=>80,
            'mandatory'=>true
        ],
        'country' => [
            'type' => 'string',
            'min'=>3,
            'max'=>3,
        ],
        'postalCode' => [
            'type' => 'string',
            'min'=>3,
            'max'=>10,
            'mandatory'=>true
        ]
    ];
}

class DHL_Details  extends DHL_basic
{
    public static $propertyTypes = [
        'dim' => [
            'type' => 'DHL_Dim',
        ],
        'weight' => [
            'type' => 'DHL_Weight',
            'mandatory'=>true
        ]
    ];
}


class DHL_consignee  extends DHL_basic
{
    public static $propertyTypes = [
        'ContactAddress' => [
            'type' => 'DHL_FullAdress',
        ],
        'Locker' => [
            'type' => 'DHL_Locker',
        ],
        'PostOffice' => [
            'type' => 'DHL_PostOffice',
        ],
        'POBox' => [
            'type' => 'DHL_POBox',
        ]
    ];
}


class DHL_Paket_DE_Versenden_REST_API {
    var $configuration, $env;
    public const EKP = MODULE_SHIPPING_DHLGKAPI_EKP;
    public const EXPORT_TYPES = [
        "OTHER","PRESENT","COMMERCIAL_SAMPLE","DOCUMENT","RETURN_OF_GOODS","COMMERCIAL_GOODS"
    ];
    public const CUSTOM_SHIPPING_CONDITIONS = [
        "DDU","DAP","DDP","DDX","DXV"
    ];
    public const API_URL = [
        'sandbox'=>'https://api-sandbox.dhl.com/parcel/de/shipping/v2/',
        'production'=>'https://api-eu.dhl.com/parcel/de/shipping/v2/',
    ];
    private const APP_ID = 'WTsojfzEAYw2p8BYwNBh0AFIBuK7uMaw';
    private const APP_SECRET = 'WPa10ZymGzzTAETu';
    public const LABEL_FORMATS = [
        "A4" => 'Common Label Laserdruck A4 Normalpapier',
        "910-300-700" => 'Common Label Laserdruck (Bogen A5) 105x208mm (910-300-700)',
        "910-300-700-oz" => 'Common Label Laserdruck (Bogen A5) 105x208mm (910-300-700) ohne Zusatzetiketten',
        "910-300-710" => 'Common Label Laserdruck 105x209mm (910-300-710)',
        "910-300-600" => 'Common Label Thermodruck (Faltband) 103x199mm (910-300-600)',
        "910-300-610" => 'Common Label Thermodruck (Rolle) 103x199mm (910-300-610)',
        "910-300-400" => 'Common Label Thermodruck (Faltband) 103x150mm (910-300-400)',
        "910-300-410" => 'Common Label Thermodruck (Rolle) 103x150mm (910-300-410)',
        "910-300-300" => 'Common Label Laserdruck (Bogen A5) 105x148mm (910-300-300)',
        "910-300-300-oz" => 'Common Label Laserdruck (Bogen A5) 105x148mm (910-300-300) ohne Zusatzetiketten'
    ];
    public const LIMITS_SPECIALS = [
        1=>'Maximal Länge für Sperrgutsendungen 200cm',
        2=>'Maximal Länge für Sperrgutsendungen bis zu 200cm, abhängig vom Zielland',
        3=>'Maximalgewicht abghängig vom Zielland'
    ];
    public const PRODUKTE = [
        'DHL_Paket'=>[
            'SHORT_TITLE'=>'National (ohne Services)',
            'description'=>'Paket im nationalen Bereich (ohne Services)',
            'Verfahren'=>'01',
            'Teilnahme'=>'01',
            'id'=>'NAT_NOSERV',
            'GKV_API'=>'V01PAK',
            'LIMITS'=>[
                'Min'=>[
                    'length'=>15,
                    'width'=>11,
                    'height'=>1,
                    'weight'=>.01
                ],
                'Max'=>[
                    'length'=>120,
                    'length_special'=>1,
                    'width'=>60,
                    'width_special'=>1,
                    'height'=>60,
                    'height_special'=>1,
                    'weight'=>31.5
                ]
            ],
            'Services'=>[
                'premium'=>false,
                'economy'=>false,
                'closestDropPoint'=>false,
                'preferredNeighbour'=>false,
                'preferredLocation'=>false,
                'visualCheckOfAge'=>false,
                'namedPersonOnly'=>false,
                'identCheck'=>false,
                'signedForByRecipient'=>false,
                'noNeighbourDelivery'=>false,
                'preferredDay'=>false,
                'endorsement'=>false, // vorausverfuegung
                'gogreen'=>false,
                'additionalInsurance'=>false,
                'bulkyGoods'=>false,
                'cashOnDelivery'=>false,
                'parcelOutletRouting'=>false,// filialrouting
                'postalDeliveryDutyPaid'=>false,
                'dhlRetoure'=>true, // beileger
            ]
        ],
        'DHL_Paket_Services'=>[
            'SHORT_TITLE'=>'National (mit Services)',
            'description'=>'Paket im nationalen Bereich (mit Services)',
            'Verfahren'=>'01',
            'Teilnahme'=>'02',
            'id'=>'NAT_SERV',
            'GKV_API'=>'V01PAK',
            'LIMITS'=>[
                'Min'=>[
                    'length'=>15,
                    'width'=>11,
                    'height'=>1,
                    'weight'=>.01
                ],
                'Max'=>[
                    'length'=>120,
                    'length_special'=>1,
                    'width'=>60,
                    'width_special'=>1,
                    'height'=>60,
                    'height_special'=>1,
                    'weight'=>31.5
                ]
            ],
            'Services'=>[
                'premium'=>false,
                'economy'=>false,
                'closestDropPoint'=>false,
                'preferredNeighbour'=>true,
                'preferredLocation'=>true,
                'visualCheckOfAge'=>true,
                'namedPersonOnly'=>true,
                'identCheck'=>true,
                'signedForByRecipient'=>true,
                'noNeighbourDelivery'=>true,
                'preferredDay'=>true,
                'endorsement'=>false, // vorausverfuegung
                'gogreen'=>false,
                'additionalInsurance'=>true,
                'bulkyGoods'=>true,
                'cashOnDelivery'=>true,
                'parcelOutletRouting'=>true,// filialrouting
                'postalDeliveryDutyPaid'=>false,
                'dhlRetoure'=>true, // beileger
            ]
        ],
        'DHL_Paket_GoGreen'=>[
            'SHORT_TITLE'=>'National (GoGreen)',
            'description'=>'Paket im nationalen Bereich mit GoGreen',
            'Verfahren'=>'01',
            'Teilnahme'=>'03',
            'GKV_API'=>'V01PAK',
            'id'=>'NAT_GOGREEN',
            'LIMITS'=>[
                'Min'=>[
                    'length'=>15,
                    'width'=>11,
                    'height'=>1,
                    'weight'=>.01
                ],
                'Max'=>[
                    'length'=>120,
                    'length_special'=>1,
                    'width'=>60,
                    'width_special'=>1,
                    'height'=>60,
                    'height_special'=>1,
                    'weight'=>31.5
                ]
            ],
            'Services'=>[
                'premium'=>false,
                'economy'=>false,
                'closestDropPoint'=>false,
                'preferredNeighbour'=>true,
                'preferredLocation'=>true,
                'visualCheckOfAge'=>true,
                'namedPersonOnly'=>true,
                'identCheck'=>true,
                'signedForByRecipient'=>true,
                'noNeighbourDelivery'=>true,
                'preferredDay'=>true,
                'endorsement'=>false, // vorausverfuegung
                'gogreen'=>true,
                'additionalInsurance'=>true,
                'bulkyGoods'=>true,
                'cashOnDelivery'=>true,
                'parcelOutletRouting'=>true,// filialrouting
                'postalDeliveryDutyPaid'=>false,
                'dhlRetoure'=>true, // beileger
            ]
        ],
        'DHL_Paket_International (ohne GoGreen)'=>[
            'SHORT_TITLE'=>'B2C Weltweit',
            'description'=>'Business-To-Consumer Versand weltweit',
            'Verfahren'=>'53',
            'Teilnahme'=>'01',
            'id'=>'INT',
            'GKV_API'=>'V53WPAK',
            'LIMITS'=>[
                'Min'=>[
                    'length'=>15,
                    'width'=>11,
                    'height'=>1,
                    'weight'=>.01
                ],
                'Max'=>[
                    'length'=>120,
                    'length_special'=>2,
                    'width'=>60,
                    'width_special'=>2,
                    'height'=>60,
                    'height_special'=>2,
                    'weight'=>31.5,
                    'weight_special'=>3
                ]
            ],
            'Services'=>[
                'premium'=>true,
                'economy'=>false,
                'closestDropPoint'=>false,
                'preferredNeighbour'=>false,
                'preferredLocation'=>false,
                'visualCheckOfAge'=>false,
                'namedPersonOnly'=>false,
                'identCheck'=>false,
                'signedForByRecipient'=>false,
                'noNeighbourDelivery'=>false,
                'preferredDay'=>false,
                'endorsement'=>true, // vorausverfuegung
                'gogreen'=>false,
                'additionalInsurance'=>true,
                'bulkyGoods'=>true,
                'cashOnDelivery'=>true,
                'parcelOutletRouting'=>false,// filialrouting
                'postalDeliveryDutyPaid'=>true,
                'dhlRetoure'=>false, // beileger
            ]
        ],
        'DHL_Paket_International_GoGreen'=>[
            'SHORT_TITLE'=>'B2C Weltweit (GoGreen)',
            'description'=>'Business-To-Consumer Versand weltweit (mit GoGreen)',
            'Verfahren'=>'53',
            'Teilnahme'=>'02',
            'id'=>'INT_GOGREEN',
            'GKV_API'=>'V53WPAK',
            'LIMITS'=>[
                'Min'=>[
                    'length'=>15,
                    'width'=>11,
                    'height'=>1,
                    'weight'=>.01
                ],
                'Max'=>[
                    'length'=>120,
                    'length_special'=>2,
                    'width'=>60,
                    'width_special'=>2,
                    'height'=>60,
                    'height_special'=>2,
                    'weight'=>31.5,
                    'weight_special'=>3
                ]
            ],
            'Services'=>[
                'premium'=>true,
                'economy'=>false,
                'closestDropPoint'=>false,
                'preferredNeighbour'=>false,
                'preferredLocation'=>false,
                'visualCheckOfAge'=>false,
                'namedPersonOnly'=>false,
                'identCheck'=>false,
                'signedForByRecipient'=>false,
                'noNeighbourDelivery'=>false,
                'preferredDay'=>false,
                'endorsement'=>true, // vorausverfuegung
                'gogreen'=>true,
                'additionalInsurance'=>true,
                'bulkyGoods'=>true,
                'cashOnDelivery'=>true,
                'parcelOutletRouting'=>false,// filialrouting
                'postalDeliveryDutyPaid'=>true,
                'dhlRetoure'=>false, // beileger
            ]
        ],
        'DHL_Europaket'=>[
            'SHORT_TITLE'=>'B2C EU',
            'description'=>'Business-To-Business Versand, Nutzung für Versand '
                .'an Endkunden nicht gestattet, überwiegend in EU-Länder unter '
                .'Nutzung des DHL-eigenen Netzes.',
            'Verfahren'=>'54',
            'Teilnahme'=>'01',
            'id'=>'EU',
            'GKV_API'=>'V54EPAK',
            'LIMITS'=>[
                'Min'=>[
                    'length'=>15,
                    'width'=>11,
                    'height'=>3.5,
                    'weight'=>.01
                ],
                'Max'=>[
                    'length'=>120,
                    'width'=>60,
                    'height'=>60,
                    'weight'=>31.5
                ]
            ],
            'Services'=>[
                'premium'=>false,
                'economy'=>false,
                'closestDropPoint'=>false,
                'preferredNeighbour'=>false,
                'preferredLocation'=>false,
                'visualCheckOfAge'=>false,
                'namedPersonOnly'=>false,
                'identCheck'=>false,
                'signedForByRecipient'=>false,
                'noNeighbourDelivery'=>false,
                'preferredDay'=>false,
                'endorsement'=>false, // vorausverfuegung
                'gogreen'=>false,
                'additionalInsurance'=>true,
                'bulkyGoods'=>false,
                'cashOnDelivery'=>false,
                'parcelOutletRouting'=>false,// filialrouting
                'postalDeliveryDutyPaid'=>false,
                'dhlRetoure'=>false, // beileger
            ]
        ],
        'DHL_Europaket_GoGreen'=>[
            'SHORT_TITLE'=>'B2C EU (GoGreen)',
            'description'=>'Business-To-Business Versand (mit GoGreen), Nutzung für Versand '
                .'an Endkunden nicht gestattet, überwiegend in EU-Länder unter '
                .'Nutzung des DHL-eigenen Netzes',
            'Verfahren'=>'54',
            'Teilnahme'=>'02',
            'id'=>'EU_GOGREEN',
            'GKV_API'=>'V54EPAK',
            'LIMITS'=>[
                'Min'=>[
                    'length'=>15,
                    'width'=>11,
                    'height'=>3.5,
                    'weight'=>.01
                ],
                'Max'=>[
                    'length'=>120,
                    'width'=>60,
                    'height'=>60,
                    'weight'=>31.5
                ]
            ],
            'Services'=>[
                'premium'=>false,
                'economy'=>false,
                'closestDropPoint'=>false,
                'preferredNeighbour'=>false,
                'preferredLocation'=>false,
                'visualCheckOfAge'=>false,
                'namedPersonOnly'=>false,
                'identCheck'=>false,
                'signedForByRecipient'=>false,
                'noNeighbourDelivery'=>false,
                'preferredDay'=>false,
                'endorsement'=>false, // vorausverfuegung
                'gogreen'=>true,
                'additionalInsurance'=>true,
                'bulkyGoods'=>false,
                'cashOnDelivery'=>false,
                'parcelOutletRouting'=>false,// filialrouting
                'postalDeliveryDutyPaid'=>false,
                'dhlRetoure'=>false, // beileger
            ]
        ],
        'Kleinpaket'=>[
            'SHORT_TITLE'=>'Kleinpaket National',
            'description'=>'nationaler Versand für kleine Sendungen',
            'Verfahren'=>'62',
            'Teilnahme'=>'01',
            'id'=>'KLP',
            'GKV_API'=>'V62KP',
            'LIMITS'=>[
                'Min'=>[
                    'length'=>10,
                    'width'=>7,
                    'height'=>.1,
                    'weight'=>.01
                ],
                'Max'=>[
                    'length'=>35,
                    'width'=>25,
                    'height'=>8,
                    'weight'=>1
                ]
            ],
            'Services'=>[
                'premium'=>false,
                'economy'=>false,
                'closestDropPoint'=>false,
            'preferredNeighbour'=>true,
            'preferredLocation'=>true,
                'visualCheckOfAge'=>false,
                'namedPersonOnly'=>false,
                'identCheck'=>false,
                'signedForByRecipient'=>false,
                'noNeighbourDelivery'=>false,
                'preferredDay'=>false,
                'endorsement'=>false, // vorausverfuegung
            'gogreen'=>true,
                'additionalInsurance'=>false,
                'bulkyGoods'=>false,
                'cashOnDelivery'=>false,
            'parcelOutletRouting'=>true,// filialrouting
                'postalDeliveryDutyPaid'=>false,
            'dhlRetoure'=>true, // beileger
            ],
        ],
        'Warenpost'=>[
            'SHORT_TITLE'=>'Warenpost National',
            'description'=>'nationaler Versand für kleine Sendungen im Briefnetz'
                .' der deutschen Post',
            'Verfahren'=>'62',
            'Teilnahme'=>'01',
            'id'=>'WAR',
            'GKV_API'=>'V62WP',
            'LIMITS'=>[
                'Min'=>[
                    'length'=>10,
                    'width'=>7,
                    'height'=>.1,
                    'weight'=>.01
                ],
                'Max'=>[
                    'length'=>35,
                    'width'=>25,
                    'height'=>5,
                    'weight'=>1
                ]
            ],
            'Services'=>[
                'premium'=>false,
                'economy'=>false,
                'closestDropPoint'=>false,
                'preferredNeighbour'=>true,
                'preferredLocation'=>true,
                'visualCheckOfAge'=>false,
                'namedPersonOnly'=>false,
                'identCheck'=>false,
                'signedForByRecipient'=>false,
                'noNeighbourDelivery'=>false,
                'preferredDay'=>false,
                'endorsement'=>false, // vorausverfuegung
                'gogreen'=>false,
                'additionalInsurance'=>false,
                'bulkyGoods'=>false,
                'cashOnDelivery'=>false,
                'parcelOutletRouting'=>true,// filialrouting
                'postalDeliveryDutyPaid'=>false,
                'dhlRetoure'=>true, // beileger
            ],
        ],
        'Warenpost_GoGreen'=>[
            'SHORT_TITLE'=>'Warenpost National (GoGreen)',
            'description'=>'nationaler Versand für kleine Sendungen im Briefnetz'
                .' der deutschen Post (mit GoGreen)',
            'Verfahren'=>'62',
            'Teilnahme'=>'02',
            'id'=>'WAR_GOGREEN',
            'GKV_API'=>'V62WP',
            'LIMITS'=>[
                'Min'=>[
                    'length'=>10,
                    'width'=>7,
                    'height'=>.1,
                    'weight'=>.01
                ],
                'Max'=>[
                    'length'=>35,
                    'width'=>25,
                    'height'=>5,
                    'weight'=>1
                ]
            ],
            'Services'=>[
                'premium'=>false,
                'economy'=>false,
                'closestDropPoint'=>false,
                'preferredNeighbour'=>true,
                'preferredLocation'=>true,
                'visualCheckOfAge'=>false,
                'namedPersonOnly'=>false,
                'identCheck'=>false,
                'signedForByRecipient'=>false,
                'noNeighbourDelivery'=>false,
                'preferredDay'=>false,
                'endorsement'=>false, // vorausverfuegung
                'gogreen'=>true,
                'additionalInsurance'=>false,
                'bulkyGoods'=>false,
                'cashOnDelivery'=>false,
                'parcelOutletRouting'=>true,// filialrouting
                'postalDeliveryDutyPaid'=>false,
                'dhlRetoure'=>true, // beileger
            ],
        ],
        'Warenpost International'=>[
            'SHORT_TITLE'=>'Warenpost International',
            'description'=>'Internationaler Versand für kleine Sendungen im '
                .' Briefnetz der Postgesellschaften',
            'Verfahren'=>'66',
            'Teilnahme'=>'01',
            'id'=>'WAR_INT',
            'GKV_API'=>'V66WPI',
            'LIMITS'=>[
                'Min'=>[
                    'length'=>14,
                    'width'=>9,
                    'height'=>.1,
                    'weight'=>.01
                ],
                'Max'=>[
                    'length'=>35.3,
                    'width'=>25,
                    'height'=>10,
                    'weight'=>1
                ]
            ],
            'Services'=>[
                'premium'=>true,
                'economy'=>false,
                'closestDropPoint'=>false,
                'preferredNeighbour'=>false,
                'preferredLocation'=>false,
                'visualCheckOfAge'=>false,
                'namedPersonOnly'=>false,
                'identCheck'=>false,
                'signedForByRecipient'=>false,
                'noNeighbourDelivery'=>false,
                'preferredDay'=>false,
                'endorsement'=>false, // vorausverfuegung
                'gogreen'=>false,
                'additionalInsurance'=>false,
                'bulkyGoods'=>false,
                'cashOnDelivery'=>false,
                'parcelOutletRouting'=>false,// filialrouting
                'postalDeliveryDutyPaid'=>false,
                'dhlRetoure'=>false, // beileger
            ],
        ],
        'Warenpost International_Premium'=>[
            'SHORT_TITLE'=>'Warenpost International Premium',
            'description'=>'Internationaler Versand (Premium Versand Pflicht) für kleine Sendungen im '
                .' Briefnetz der Postgesellschaften',
            'Verfahren'=>'66',
            'Teilnahme'=>'02',
            'id'=>'WAR_INT_PREMIUM',
            'GKV_API'=>'V66WPI',
            'LIMITS'=>[
                'Min'=>[
                    'length'=>14,
                    'width'=>9,
                    'height'=>.1,
                    'weight'=>.01
                ],
                'Max'=>[
                    'length'=>35.3,
                    'width'=>25,
                    'height'=>10,
                    'weight'=>1
                ]
            ],
            'Services'=>[
                'premium'=>true,
                'economy'=>false,
                'closestDropPoint'=>false,
                'preferredNeighbour'=>false,
                'preferredLocation'=>false,
                'visualCheckOfAge'=>false,
                'namedPersonOnly'=>false,
                'identCheck'=>false,
                'signedForByRecipient'=>false,
                'noNeighbourDelivery'=>false,
                'preferredDay'=>false,
                'endorsement'=>false, // vorausverfuegung
                'gogreen'=>false,
                'additionalInsurance'=>false,
                'bulkyGoods'=>false,
                'cashOnDelivery'=>false,
                'parcelOutletRouting'=>false,// filialrouting
                'postalDeliveryDutyPaid'=>false,
                'dhlRetoure'=>false, // beileger
            ],
        ],
        'Warenpost International_OhneServices'=>[
            'SHORT_TITLE'=>'Warenpost International (ohne Services)',
            'description'=>'Internationaler Versand (ohne Services) für kleine Sendungen im '
                .' Briefnetz der Postgesellschaften',
            'Verfahren'=>'66',
            'Teilnahme'=>'03',
            'id'=>'WAR_INT_NOSERV',
            'GKV_API'=>'V66WPI',
            'LIMITS'=>[
                'Min'=>[
                    'length'=>14,
                    'width'=>9,
                    'height'=>.1,
                    'weight'=>.01
                ],
                'Max'=>[
                    'length'=>35.3,
                    'width'=>25,
                    'height'=>10,
                    'weight'=>1
                ]
            ],
            'Services'=>[
                'premium'=>false,
                'economy'=>false,
                'closestDropPoint'=>false,
                'preferredNeighbour'=>false,
                'preferredLocation'=>false,
                'visualCheckOfAge'=>false,
                'namedPersonOnly'=>false,
                'identCheck'=>false,
                'signedForByRecipient'=>false,
                'noNeighbourDelivery'=>false,
                'preferredDay'=>false,
                'endorsement'=>false, // vorausverfuegung
                'gogreen'=>false,
                'additionalInsurance'=>false,
                'bulkyGoods'=>false,
                'cashOnDelivery'=>false,
                'parcelOutletRouting'=>false,// filialrouting
                'postalDeliveryDutyPaid'=>false,
                'dhlRetoure'=>false, // beileger
            ],
        ],
        'Warenpost International_GoGreen'=>[
            'SHORT_TITLE'=>'Warenpost International (GoGreen)',
            'description'=>'Internationaler Versand (mit GoGreen) für kleine Sendungen im '
                .' Briefnetz der Postgesellschaften',
            'Verfahren'=>'66',
            'Teilnahme'=>'04',
            'id'=>'WAR_INT_GOGREEN',
            'GKV_API'=>'V66WPI',
            'LIMITS'=>[
                'Min'=>[
                    'length'=>14,
                    'width'=>9,
                    'height'=>.1,
                    'weight'=>.01
                ],
                'Max'=>[
                    'length'=>35.3,
                    'width'=>25,
                    'height'=>10,
                    'weight'=>1
                ]
            ],
            'Services'=>[
                'premium'=>true,
                'economy'=>false,
                'closestDropPoint'=>false,
                'preferredNeighbour'=>false,
                'preferredLocation'=>false,
                'visualCheckOfAge'=>false,
                'namedPersonOnly'=>false,
                'identCheck'=>false,
                'signedForByRecipient'=>false,
                'noNeighbourDelivery'=>false,
                'preferredDay'=>false,
                'endorsement'=>false, // vorausverfuegung
                'gogreen'=>true,
                'additionalInsurance'=>false,
                'bulkyGoods'=>false,
                'cashOnDelivery'=>false,
                'parcelOutletRouting'=>false,// filialrouting
                'postalDeliveryDutyPaid'=>false,
                'dhlRetoure'=>false, // beileger
            ],
        ],
        
    ];

    public function get_auth(){
        return base64_encode($this->configuration['auth'][$this->env]['user'].':'.$this->configuration['auth'][$this->env]['password']);
    }
    
    public function __construct(){
        $this->env = (MODULE_SHIPPING_DHLGKAPI_USER=='user-valid') ? 'sandbox' : 'production';
        
        $this->configuration = [
            'auth'=>[
                $this->env=>[
                    'user'=>MODULE_SHIPPING_DHLGKAPI_USER,
                    'password'=>MODULE_SHIPPING_DHLGKAPI_PASSWORD,
                ]
            ]
        ];
        
        
    }

    /**
     * Manifestieren bezeichnet die endgültige Übergabe der Sendungsdaten an 
     * die Post & DHL Backend-Systeme. 
     * Dies geschieht im Rahmen des sogenannten Tagesabschlusses. Bei Sendungen, 
     * die per Webservice angelegt wurden, erfolgt der Tagesabschluss entweder 
     * automatisch am Tagesende zu einem festgelegten Zeitpunkt oder durch die 
     * Methode "doManifest".
     */
    public function Manifest(){
        
    }

    public function createAPIKey(){
        //$APP_ID = DHL_VERSENDEN_APPLICATION_ID;
        return self::curl_call();
    }
    
    /**
     * Testfunktion
     * @return string response json codiert
     */
    public function curl_call(){
        $ch = curl_init(self::API_URL[$this->env]);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Accept: application/json',
            'dhl-api-key: '.self::APP_ID,
        ]);
        curl_setopt($ch, CURLOPT_HEADER, false); // header rueckgabe
        curl_setopt($ch, CURLOPT_USERAGENT, 'YES4Trade');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        // grab URL and pass it to the browser
        $response = curl_exec($ch);
        // close cURL resource, and free up system resources
        curl_close($ch);
        return $response;
    }
    
    /**
     * Fuehrt einen Call in der DHL Api aus
     * 
     * @param string $url
     * @param string $postFields (json dekodierte POST Parameter)
     * @return array [ACK="Error|Success","RESPONSE"=>json response]
     * @throws Exception
     */
    public function call( $url, $postFields = null, bool $delete = false){
        $curl = curl_init();
        $curl_header = [
            CURLOPT_USERAGENT => 'YES4Trade',
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => "",
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_HTTPHEADER => [
                'accept: application/json',
                "Accept-Language: de-DE",
                "Authorization: Basic ".$this->get_auth(),
                "dhl-api-key: ".self::APP_ID,
                "content-type: application/json"
            ]
        ];
        if($postFields !== null){
            $curl_header[CURLOPT_CUSTOMREQUEST] = "POST";
            $curl_header[CURLOPT_POSTFIELDS] = $postFields;
            if(isset($_GET['debug'])){
                yes_debug($postFields);
            }
        }
        if($delete === true){
            $curl_header[CURLOPT_CUSTOMREQUEST] = "DELETE";
        }
        if(main::isDev(true)){ // nichts relevantes
            $curl_header[CURLINFO_HEADER_OUT] = true;
        }
        curl_setopt_array($curl, $curl_header);
        $response = curl_exec($curl);
        if(main::isDev(true)){  // nichts relevantes
            $headerSent = curl_getinfo($curl, CURLINFO_HEADER_OUT );
        }
        $err = curl_error($curl);
        curl_close($curl);
        if ($err) {
                throw new Exception('DHL Call Error: '.$err);
        }
        $error = false;
        $chk = [];
        try{
            $chk = self::check_response_errors( $response );
        }catch( Exception $e){
        /*if(self::check_response_errors( $response )){
        }*/
            $error = true;
        }
        return [
            'ERROR'=>$error,
            'WARNINGS'=>$chk,
            'RESPONSE'=>$response
        ];
    }
    
    public function getManifests(){
        $url = self::API_URL[ $this->env ].'manifests?date='.date('Y-m-d');
        return $this->call( $url );
    }
    
    public function createShipmentOrder( array $shipments, bool $validate = false ){
        $url = self::API_URL[ $this->env ].'orders';
        if($validate){
            $url .= '?validate=true';
        }
        $requestObjectFull = $this->getShipmentDataArray($shipments);
        return $this->call( $url, json_encode($requestObjectFull) );
    }
    
    public function deleteShipmentOrder( string $trackingno ){
        $url = self::API_URL[ $this->env ].'orders';
        $url .= '?profile=STANDARD_GRUPPENPROFIL&shipment='.$trackingno;
        return $this->call( $url,null,true );
    }
    
    private static function check_response_errors( $response ){
        $json = json_decode($response);
        $warnings = [];
        if(gettype($json->status) == 'object'){
            if($json->status->statusCode == 200){
                return $warnings;
            }
            $errors = [];
            $warnings = [];
            foreach($json->items as $jit){
                foreach($jit->validationMessages as $vm){
                    switch($vm->validationState){
                        case 'Error':
                            $errors[] = sprintf(
                                "Error [%s]: %s",
                                $vm->property,
                                $vm->validationMessage
                            );
                            break;
                        case 'Warning':
                            $warnings[] = sprintf(
                                "Warning [%s]: %s",
                                $vm->property,
                                $vm->validationMessage
                            );
                            break;
                    }
                }
            }
            if(sizeOf($errors)){
                throw new Exception(sprintf(
                    "%s / Status #%d\n<i>%s</i><br />%s<br />%s",
                    $json->status->title,$json->status->statusCode,$json->status->detail,
                    implode('<br />',$errors),
                    implode('<br />',$warnings)
                ));
            }elseif(sizeOf($warnings)){
                $warnings[] = sprintf(
                    "%s / Status #%d\n<i>%s</i><br />%s",
                    $json->status->title,$json->status->statusCode,$json->status->detail,
                    implode('<br />',$warnings)
                );
            }
            
        }elseif($json->status != 200){
            throw new Exception(sprintf(
                    "DHL API Error #%d\n%s\n<i>%s</i>",
                    $json->status,$json->title,$json->detail
            ));
        }
        return $warnings;
    }
    
    public function getShipmentDataArray(array $data){
        return [
            'profile'=>'STANDARD_GRUPPENPROFIL',
            'shipments'=>$data
        ];
    }
    
    public static function get_consignee( array $address ){
        $suburb = $address['suburb'];
        $street_address = $address['street_address'];
        if (preg_match("/[0-9]{6}/", $suburb) or preg_match("/Postfiliale/i", $street_address) or preg_match("/Packstation/i", $street_address)) {
            //Packstation
            if (preg_match("/Packstation/i", $street_address)) {
                $DHL_consignee = new DHL_consignee([
                    'locker'=>new DHL_Locker([
                        'name'=>'Packstation',
                        'lockerID'=>preg_replace('/[^0-9]/','',$street_address),
                        'postNumber'=>preg_replace('/[^0-9]/','',$suburb),
                        'city'=>$address['city'],
                        'country'=>$address['country'],
                        'postalCode'=>$address['zip'],
                     ])
                ]);
            }
            //Postfiliale
            if (preg_match("/Postfiliale/i", $street_address)) {
                $DHL_consignee = new DHL_consignee([
                    'PostOffice'=>new DHL_PostOffice([
                        'name'=>'Postfiliale',
                        'retailID'=>preg_replace('/[^0-9]/','',$street_address),
                        'postNumber'=>preg_replace('/[^0-9]/','',$suburb),
                        'city'=>$address['city'],
                        'country'=>$address['country'],
                        'postalCode'=>$address['zip'],
                        'email'=>$address['email'],
                     ])
                ]);
            }
        }else{
            if(isset($address['name']) or isset($address['company'])){
                // from order/rekla
                $name1 = (isset($address['company']) and !empty($address['company'])) ? $address['company'] : $address['name'];
                $name2 = (isset($address['company']) and !empty($address['company'])) ? $address['name'] : '';
                $street = $address['street_address'];
                $postalCode = $address['zip'];
            }else{
                $name1 = $address['name1'];
                $name2 = $address['name2'];
                $street = $address['addressStreet'];
                $postalCode = $address['postalCode'];
            }
            $address_data = [
                    'name1'=>$name1
            ];
            if(!empty($name2)){
                    $address_data['name2'] = $name2;
            }
            if(!empty($name3)){
                    $address_data['name3'] = $name3;
            }
            if(isset($address['suburb']) and !empty($address['suburb'])){
                $address_data['name3'] = $address['suburb'];
            }
            $address_data['addressStreet'] = $street;
            $address_data['city'] = $address['city'];
            $address_data['country'] = $address['country'];
            $address_data['postalCode'] = $postalCode;

            if(isset($address['additionalAddressInformation1']) and !empty($address['additionalAddressInformation1'])){
                $address_data['additionalAddressInformation1'] = $address['additionalAddressInformation1'];
            }
            if(!empty($address['email'])){
                $address_data['email'] = $address['email'];
            }
            $DHL_consignee = new DHL_FullAdress($address_data);
        }
        return $DHL_consignee;
    }
    
    public static function get_shipper(){
        return new DHL_Shipper([
            'name1'=>MODULE_SHIPPING_DHLGKAPI_SHIPPER_NAME,
            'addressStreet'=>MODULE_SHIPPING_DHLGKAPI_SHIPPER_STREETNAME,
            'addressHouse'=>MODULE_SHIPPING_DHLGKAPI_SHIPPER_STREETNUMBER,
            'postalCode'=>MODULE_SHIPPING_DHLGKAPI_SHIPPER_ZIP,
            'city'=>MODULE_SHIPPING_DHLGKAPI_SHIPPER_CITY,
            'country'=>MODULE_SHIPPING_DHLGKAPI_SHIPPER_COUNTRY
        ]);       
    }
    
    
    /**
     * Consignee Adressdatensatz aus order oder reclamation
     * @param object $source_object  reclamation oder order
     * 
     */
    public static function get_consignee_data_from_source_object($source_object){
        $iso3 = yes_get_iso_code3($source_object->delivery['country']);
        if($source_object->delivery['company'] != ''){
            $name1 = $source_object->delivery['company'];
            $name2 = $source_object->delivery['name'];
        }else{
            $name1 = $source_object->delivery['name'];
            $name2 = '';
        }
        $name3 = '';
        if(!empty($source_object->delivery['suburb'])){
            $name3 = $source_object->delivery['suburb'];
        }
        if(stripos($source_object->delivery['street_address'],'Packstation')>-1){
            $name2 = $source_object->delivery['suburb'];
        }
        $address = [
            'name1'=>$name1,
            'addressStreet'=>$source_object->delivery['street_address'],
            'city'=>$source_object->delivery['city'],
            'postalCode'=>$source_object->delivery['postcode'],
            'country'=>$iso3
        ];
        if(!empty($name2)){
            $address['name2'] = $name2;
        }
        if(!empty($name3)){
            $address['name3'] = $name3;
        }
        $em = ($source_object->delivery['email_address'] != '') ? $source_object->delivery['email_address'] : $source_object->customer['email_address'];
        if(DSGVO_ALLOW_SEND_CUSTOMER_EMAIL_ADDRESS_TO_SHIPPER == 'True' or preg_match("/Postfiliale/i", $source_object->delivery['street_address'])){
            $address['email'] = $em;
        }
        return $address;
    }
    
    /**
     * dhlRetoure billingNumber ermitteln
     * @return string
     */
    public static function get_dhlRetoure_billingNumber( $produkt, $gogreen = false ){
        if(!$go_green){
            return self::EKP.'0701';
        }
        return self::EKP.'0702';
    }
    
    /**
     * shipment billingNumber ermitteln
     * Es wird zb V01PAK0101 als produkt uebergeben
     * @param string produkt
     * @return string
     */
    public static function get_billingNumber( $produkt ){
        return self::EKP.substr($produkt,-4);
    }
    
    public static function isJson($string) {
        json_decode($string);
        return json_last_error() === JSON_ERROR_NONE;
    }

    
    public static function get_from_shipping_item(yes_shipping $yes_shipping ){
        $total = 0;
        $refNo = '';
        $rID = false;
        $oID = false;
        if($yes_shipping->getReclamations_id() > 0){
            $rID = $yes_shipping->getReclamations_id();
            $reclamation = new reclamation($rID);
            $address = self::get_consignee_data_from_source_object($reclamation);
            $refNo .= 'R'.$rID;
        }else{
            $oID = $yes_shipping->getOrders_id();
            $refNo .= $oID;
            $order = new order($oID);
            $address = self::get_consignee_data_from_source_object($order);
            $shipping_costs = 0; // fuer exportDocument
            foreach($order->totals as $total){
                switch($total['class']){
                    case 'ot_shipping':
                        $shipping_costs = $total['value'];
                        break;
                    case 'ot_total':
                        $total = $total['value'];
                        break;
                }
            }
        }
        $refNo = self::fillRefNo($refNo);
        $add_params_raw = $yes_shipping->getAdditional_params();
        if(!self::isJson($add_params_raw)){
            $ref_str = (isset($rID) and $rID > 0) ? TEXT_RECLAMATION.' '.$rID:TEXT_ORDER.' '.$oID;
            throw new Exception(sprintf(
                    'Versanddatensatz #%d (%s) hat ungültige DHL Parameter: %s',
                    $yes_shipping->getId(),
                    $ref_str,
                    $add_params_raw
            ));
        }
        if(!empty($add_params_raw)){
            $add_params = json_decode($add_params_raw);
            

            // HIER SAMMELN WIR ALLE AKTUELLEN SERVICES UM DIESE DANN BEIM DHL_Services
            // INSTANTIIEREN HINZUZUFUEGEN
            $services = []; 
            $refNo = ($yes_shipping->getReclamations_id() > 0)?'R'.$yes_shipping->getReclamations_id():"".$yes_shipping->getOrders_id();
            $refNo = self::fillRefNo($refNo);
            $paketklasse = $add_params->product;
	    if(!isset($add_params->billingNumber)){
		$add_params->billingNumber = self::get_billingNumber($add_params->product.'0102');
	    }
            //$produkt = substr($paketklasse, 0, strlen($paketklasse)-4);
            foreach($add_params->services as $service=>$value){
                switch($service){
                    case 'cashOnDelivery':
                        $amount = new DHL_Service_Amount([
                            'value'=>(float)$value->amount->value,
                            'currency'=>$value->amount->currency,
                        ]);
                        $bankAccount = new DHL_Service_cashOnDelivery_bankAccount([
                            'accountHolder'=>$value->bankAccount->accountHolder,
                            'iban'=>$value->bankAccount->accountHolder
                        ]);
                        $cashOnDelivery = new DHL_Service_cashOnDelivery([
                            'amount'=>$amount,
                            'bankAccount'=>$bankAccount
                        ]);
                        $services[$service] = $cashOnDelivery;
                        break;
                    case 'bulkyGoods':
                        if($value == true){
                            $services[$service] = true;
                        }
                        break;
                    case 'visualCheckOfAge':
                        if(!empty($value)){
                            $services[$service] = $value;
                        }
                        break;
                    case 'endorsement':
                        if(!empty($value)){
                            $services[$service] = $value;
                        }
                        break;
                    case 'preferredNeighbour':
                    case 'preferredLocation':
                    case 'preferredDay':
                        if(!empty($value)){
                            $services[ $service ] = $value;
                        }
                        break;
                    case 'postalDeliveryDutyPaid':
                    case 'closestDropPoint':
                    case 'signedForByRecipient':
                    case 'namedPersonOnly':
                    case 'noNeighbourDelivery':
                    case 'premium':
                        if($value === true){
                            $services[ $service ] = $value;
                        }
                        break;
                    case 'identCheck':
                        $services[ $service ] = new DHL_Service_identCheck([
                            'firstName'=>$value->firstName,
                            'lastName'=>$value->lastName,
                        ]);
                        break;
                    case 'additionalInsurance':
                        $services[ $service ] = new DHL_Service_Amount([
                            'currency'=>$value->currency,
                            'value'=>(float)$value->value,
                        ]);
                        break;
                    case 'parcelOutletRouting':
                        if(!empty($value)){
                            $services[ $service ] = $value; // email des empfaengers
                        }
                        break;
                    case 'dhlRetoure':
                        $raddress = [
                            'name1'=>$value->returnAddress->name1,
                            'addressStreet'=>$value->returnAddress->addressStreet,
                            'postalCode'=>$value->returnAddress->postalCode,
                            'city'=>$value->returnAddress->city,
                            'country'=>$value->returnAddress->country,
                        ];
                        if(isset($value->returnAddress->additionalAddressInformation1) and !empty($value->returnAddress->additionalAddressInformation1)){
                            $raddress['additionalAddressInformation1'] = $value->returnAddress->additionalAddressInformation1;
                        }
                        $services[ $service ] = new DHL_Service_dhlRetoure([
                            'billingNumber'=>self::get_dhlRetoure_billingNumber($paketklasse),
                            'returnAddress'=>new DHL_FullAdress($raddress)
                        ]);
                        break;
                }
            }
        }
        if($add_params->customs){
            $items = [];
            foreach($yes_shipping->products as $product){
                if($yes_shipping->getReclamations_id() > 0){
                    foreach($reclamation->products as $rp){
                        if($rp['rpID'] == $product['source_products_id']){
                            $price = (float)$rp['price'];
                        }
                    }
                }else{
                    foreach($order->products as $op){
                        if($op['opID'] == $product['source_products_id']){
                            $price = (float)$op['price'];
                        }
                    }
                }
                $weight = xtc_get_ext_products_weight($product['products_id']);
                $item = new DHL_customs_item([
                    'itemDescription'=>xtc_get_products_name( $product['products_id'] ),
                    'packagedQuantity'=>(int)$product['quantity'],
                    'itemValue'=>new DHL_Service_Amount([
                        'currency'=>DEFAULT_CURRENCY,
                        'value'=>$price
                    ]),
                    'itemWeight'=>new DHL_Weight([
                        'uom'=>'kg',
                        'value'=>(int)$weight
                    ])
                ]);
                $items[] = $item;
            }
            $data = [
                'invoiceNo'=>$refNo,
                'exportType'=>$add_params->customs->exportType, // ["OTHER","PRESENT","COMMERCIAL_SAMPLE","DOCUMENT","RETURN_OF_GOODS","COMMERCIAL_GOODS"]
                'postalCharges'=>new DHL_Service_Amount([
                    'currency'=>$add_params->customs->postalCharges->currency,
                    'value'=>(float)$add_params->customs->postalCharges->value
                ]),
                'items'=>$items
            ];
            if(!empty($add_params->customs->shipperCustomsRef)){
                $data['shipperCustomsRef'] = $add_params->customs->shipperCustomsRef;
            }
            if(!empty($add_params->customs->consigneeCustomsRef)){
                $data['consigneeCustomsRef'] = $add_params->customs->consigneeCustomsRef;
            }
            if(!empty($add_params->customs->MRN)){
                $data['MPN'] = $add_params->customs->MRN;
            }
            if(!empty($add_params->customs->hasElectronicExportNotification)){
                $data['hasElectronicExportNotification'] = $add_params->customs->hasElectronicExportNotification;
            }
            if(!empty($add_params->customs->exportDescription)){
                $data['exportDescription'] = $add_params->customs->exportDescription;
            }
            if(!empty($add_params->customs->shippingConditions)){
                $data['shippingConditions'] = $add_params->customs->shippingConditions;
            }
            $customs = new DHL_customs($data);
        }
        //Gewichtsberechnung
        //Startgewicht = Versandkartongewicht
        
        // WENN IM VERSANDMODUL EIGENES GEWICHT HINTERLEGT WURDE, TROTZ
        // WEIGHT_FROM_ORDER
        if($yes_shipping->getWeight() > 0){
            // liegt in yes_shipping als 100g vor, umrechnen
            $dhl_weight = $yes_shipping->getWeight()/10;
        }else{
            $dhl_weight = SHIPPING_BOX_WEIGHT;
            if($yes_shipping->getReclamations_id() > 0){
                $dhl_weight += $reclamation->show_weight($yes_shipping->getId());
            }else{
                $dhl_weight += $order->show_weight($yes_shipping->getId());
            }
        }
        /** RAUS
        if($dhl_weight < 1){
                $dhl_weight = 1;
        }*/
        $array = [
            'product'=>$paketklasse,
            'billingNumber'=>$add_params->billingNumber,
            'refNo'=>$refNo,
            'shipDate'=>date('Y-m-d'),
            'shipper'=>self::get_shipper(),
            'consignee'=>DHL_Paket_DE_Versenden_REST_API::get_consignee($address),
            'details'=>new DHL_Details([
                'weight'=>new DHL_Weight([
                    'uom'=>'g',
                    'value'=>(int)($dhl_weight*1000)
                ])
            ]),
        ];
        if(gettype($services)!=='NULL' and is_countable($services) and sizeOf($services) ){
            $array['services'] = $services;
        }
        if(gettype($customs)!=='NULL'){
            $array['customs'] = $customs;
        }
        return new DHL_shipment($array);
    }
    

    public static function get_empty( $produkt ){
        return new DHL_shipment([
            'product'=>$produkt,
            'shipper'=>self::get_shipper()
        ]);
    }

    /**
     * funktioniert auch mit _GET
     * @param string $produkt
     * @return \DHL_shipment
     */
    public static function get_from_post( $produkt, $convert_product_value = true ){
        foreach(self::PRODUKTE as $p){
            if($p['id'] == $produkt){
                $service = $p;
            }
        }
        $services = [];
        foreach($service['Services'] as $service_name => $service_active ){
            if($service_active != 1){
                continue;
            }
            
            if(isset($_REQUEST[$service_name]) and !empty($_REQUEST[$service_name])){
                switch($service_name){
                    case 'cashOnDelivery':
                        
                        $cv = (isset($_REQUEST['cashOnDelivery_value']))?(float)str_replace(',','.',$_REQUEST['cashOnDelivery_value']):0;
                        $cc = (isset($_REQUEST['cashOnDelivery_currency']))?$_REQUEST['cashOnDelivery_currency']:'';
                        if($cv > 0 and !empty($cc)){
                            $amount = new DHL_Service_Amount([
                                'value'=>$cv,
                                'currency'=>$cc,
                            ]);
                        
                            $holder = (isset($_REQUEST['cashOnDelivery_bankAccount_accountHolder'])) ? $_REQUEST['cashOnDelivery_bankAccount_accountHolder'] : '';
                            $iban = (isset($_REQUEST['cashOnDelivery_bankAccount_iban'])) ? $_REQUEST['cashOnDelivery_bankAccount_iban'] : '';
                            $bankAccount = new DHL_Service_cashOnDelivery_bankAccount([
                                'accountHolder'=>$holder,
                                'iban'=>$iban
                            ]);
                            $cashOnDelivery = new DHL_Service_cashOnDelivery([
                                'amount'=>$amount,
                                'bankAccount'=>$bankAccount
                            ]);
                            $services[$service_name] = $cashOnDelivery;
                        }
                        break;
                    case 'bulkyGoods':
                    case 'postalDeliveryDutyPaid':
                    case 'closestDropPoint':
                    case 'signedForByRecipient':
                    case 'namedPersonOnly':
                    case 'noNeighbourDelivery':
                    case 'premium':
                        $services[$service_name] = true;
                        break;
                   
                    case 'endorsement':
                    case 'visualCheckOfAge':
                    case 'preferredNeighbour':
                    case 'preferredLocation':
                    case 'preferredDay':
                    case 'parcelOutletRouting':
                        $services[$service_name] = $_REQUEST[$service_name];
                        break;
                    case 'identCheck':
                        $fn = (isset($_REQUEST['identCheckFirstName']) and !empty($_REQUEST['identCheckFirstName'])) ? $_REQUEST['identCheckFirstName'] : '';
                        $ln = (isset($_REQUEST['identCheckLastName']) and !empty($_REQUEST['identCheckLastName'])) ? $_REQUEST['identCheckLastName'] : '';
                        if(!empty($fn) and !empty($ln)){
                            $services[ $service_name ] = new DHL_Service_identCheck([
                                'firstName'=>$fn,
                                'lastName'=>$ln,
                            ]);
                        }
                        break;
                    case 'additionalInsurance':
                        $av = (isset($_REQUEST['additionalInsurance_value']))?(float)str_replace(',','.',$_REQUEST['additionalInsurance_value']):0;
                        $ac = (isset($_REQUEST['additionalInsurance_currency']))?$_REQUEST['additionalInsurance_currency']:'';
                        if($av > 0 and !empty($ac)){
                            $services[ $service_name ] = new DHL_Service_Amount([
                                'currency'=>$ac,
                                'value'=>$av
                            ]);
                        }
                        break;
                    case 'dhlRetoure':
                        $raddress = [
                            'name1'=>(isset($_REQUEST['dhlRetoure_returnAddress_name1']))?$_REQUEST['dhlRetoure_returnAddress_name1']:MODULE_SHIPPING_DHLGKAPI_SHIPPER_NAME,
                            'addressStreet'=>(isset($_REQUEST['dhlRetoure_returnAddress_addressStreet']))?$_REQUEST['dhlRetoure_returnAddress_addressStreet']:MODULE_SHIPPING_DHLGKAPI_SHIPPER_STREETNAME.' '.MODULE_SHIPPING_DHLGKAPI_SHIPPER_STREETNUMBER,
                            'postalCode'=>(isset($_REQUEST['dhlRetoure_returnAddress_postalCode']))?$_REQUEST['dhlRetoure_returnAddress_postalCode']:MODULE_SHIPPING_DHLGKAPI_SHIPPER_ZIP,
                            'city'=>(isset($_REQUEST['dhlRetoure_returnAddress_city']))?$_REQUEST['dhlRetoure_returnAddress_city']:MODULE_SHIPPING_DHLGKAPI_SHIPPER_CITY,
                            'country'=>(isset($_REQUEST['dhlRetoure_returnAddress_country']))?$_REQUEST['dhlRetoure_returnAddress_country']:MODULE_SHIPPING_DHLGKAPI_SHIPPER_COUNTRY
                        ];
                        if(isset($_REQUEST['dhlRetoure_returnAddress_additionalAddressInformation1']) and !empty($_REQUEST['dhlRetoure_returnAddress_additionalAddressInformation1'])){
                            $raddress['additionalAddressInformation1'] = $_REQUEST['dhlRetoure_returnAddress_additionalAddressInformation1'];
                        }
                        $services[ $service_name ] = new DHL_Service_dhlRetoure([
                            'billingNumber'=>(isset($_REQUEST['dhlRetoure_billingNumber']))?$_REQUEST['dhlRetoure_billingNumber']:self::get_dhlRetoure_billingNumber($produkt),
                            'returnAddress'=>new DHL_FullAdress($raddress)
                        ]);
                        break;
                }
            }
        }
        $customs_save = false;
        if(isset($_REQUEST['customs']) and $_REQUEST['customs'] == 1){
            $customs_save = true;
            $items = [];
            $data = [
                'invoiceNo'=>(isset($_REQUEST['custom_invoiceNo']))?$_REQUEST['custom_invoiceNo']:'',
                'exportType'=>(isset($_REQUEST['custom_exportType']))?$_REQUEST['custom_exportType']:'OTHER',
                //'exportDescription'=>(isset($_REQUEST['custom_exportDescription']))?$_REQUEST['custom_exportDescription']:'',
                // HABEN WIR GAR NICHT IM CONFIG TEMPLATE
                'postalCharges'=>new DHL_Service_Amount([
                    'currency'=>(isset($_REQUEST['custom_postalCharges_currency'])) ? $_REQUEST['custom_postalCharges_currency'] : '',
                    'value'=>(isset($_REQUEST['custom_postalCharges_value']))?(float)str_replace(',','.',$_REQUEST['custom_postalCharges_value']):0.0,
                ]),
                'items'=>$items
            ];
            if(isset($_REQUEST['custom_hasElectronicExportNotification']) and $_REQUEST['custom_hasElectronicExportNotification'] == 1){
                $data['hasElectronicExportNotification'] = $_REQUEST['custom_hasElectronicExportNotification'];
            }
            if(isset($_REQUEST['custom_MRN']) and !empty($_REQUEST['custom_MRN'])){
                $data['MRN'] = $_REQUEST['custom_MRN'];
            }
            if(isset($_REQUEST['custom_shippingConditions']) and !empty($_REQUEST['custom_shippingConditions'])){
                $data['shippingConditions'] = $_REQUEST['custom_shippingConditions'];
            }
            if(isset($_REQUEST['custom_shipperCustomsRef']) and !empty($_REQUEST['custom_shipperCustomsRef'])){
                $data['shipperCustomsRef'] = $_REQUEST['custom_shipperCustomsRef'];
            }
            if(isset($_REQUEST['custom_consigneeCustomsRef']) and !empty($_REQUEST['custom_consigneeCustomsRef'])){
                $data['consigneeCustomsRef'] = $_REQUEST['custom_consigneeCustomsRef'];
            }
            $data['permitNo'] = 'approval / permit';
            $data['attestationNo'] = 'confirmation / attestation';
            
            $customs = new DHL_customs($data);
        }
        if($convert_product_value){
            //$produkt = substr($produkt,0,strlen($produkt)-4);
        }
        $dhl_produkt_info = (array)json_decode(MODULE_SHIPPING_DHLGKAPI_TEILNAHME);
        $dhl_produkt = $dhl_produkt_info[ $produkt ];
        if($customs_save){
	    $data = [
                'product'=>$dhl_produkt,
                'customs'=>$customs,
            ];
	    if(is_countable($services) and sizeOf($services)){
                $data['services']=$services;
            }
            return new DHL_shipment($data);
        }else{
		$data = [
	                'product'=>$dhl_produkt,
        	];
        	if(is_countable($services) and sizeOf($services)){
	                $data['services']=$services;
	        }
		return new DHL_shipment($data);
        }
    }
    
    public static function get_from_shipper_profile( string $profile_name ){
        
        $sp = \YES4Trade\Model\shipper_profiles::get_profile($profile_name);
        $json = json_decode($sp->versender_params);
        foreach(self::PRODUKTE as $p){
            if( $p['id'] == $json->product ){
                $service = $p;
            }
        }
        if(!$service){
            $produkt = current(self::PRODUKTE)['GKV_API'];
            $billing_str = current(self::PRODUKTE)['GKV_API'].current(self::PRODUKTE)['Verfahren'].current(self::PRODUKTE)['Teilnahme'];
            return self::get_empty($produkt);
        }
        $services = [];
        foreach($service['Services'] as $service_name => $service_active ){
            if($service_active != 1){
                continue;
            }
            switch($service_name){
                case 'cashOnDelivery':
                    if(gettype($json->services) == 'array' or gettype($json->services) == 'NULL'){
                        
                    }else{
                        if(property_exists( $json->services, 'cashOnDelivery')){
                            $cv = (float)$json->services->cashOnDelivery->amount->value;
                            $cc = $json->services->cashOnDelivery->amount->currency;
                            $amount = new DHL_Service_Amount([
                                'value'=>$cv,
                                'currency'=>$cc,
                            ]);

                            $holder = $json->services->cashOnDelivery->bankAccount->accountHolder;
                            $iban = $json->services->cashOnDelivery->bankAccount->iban;
                            $bankAccount = new DHL_Service_cashOnDelivery_bankAccount([
                                'accountHolder'=>$holder,
                                'iban'=>$iban
                            ]);
                            $cashOnDelivery = new DHL_Service_cashOnDelivery([
                                'amount'=>$amount,
                                'bankAccount'=>$bankAccount
                            ]);
                            $services[$service_name] = $cashOnDelivery;
                        }
                    }
                    break;
                case 'bulkyGoods':
                case 'postalDeliveryDutyPaid':
                case 'closestDropPoint':
                case 'signedForByRecipient':
                case 'namedPersonOnly':
                case 'noNeighbourDelivery':
                case 'premium':
                case 'endorsement':
                case 'visualCheckOfAge':
                case 'preferredNeighbour':
                case 'preferredLocation':
                case 'preferredDay':
                case 'parcelOutletRouting':
                    if(gettype($json->services) == 'array' or gettype($json->services) == 'NULL'){
                        
                    }else{
                        if(property_exists( $json->services, $service_name)){
                            $services[$service_name] = $json->services->$service_name;
                        }
                    }
                    break;
                case 'identCheck':
                    if(gettype($json->services) == 'array' or gettype($json->services) == 'NULL'){
                        
                    }else{
                        if(property_exists( $json->services, $service_name)){
                            $fn = $json->services->$service_name->firstName;
                            $ln = $json->services->$service_name->lastName;
                            $services[ $service_name ] = new DHL_Service_identCheck([
                                'firstName'=>$fn,
                                'lastName'=>$ln,
                            ]);
                        }
                    }
                    break;
                case 'additionalInsurance':
                    if(gettype($json->services) == 'array' or gettype($json->services) == 'NULL'){
                        
                    }else{
                        if(property_exists( $json->services, $service_name)){
                            $av = $json->services->$service_name->value;
                            $ac = $json->services->$service_name->currency;
                            $services[ $service_name ] = new DHL_Service_Amount([
                                'currency'=>$ac,
                                'value'=>(float)$av
                            ]);
                        }
                    }
                    break;
                case 'dhlRetoure':
                    if(gettype($json->services) == 'array' && !sizeOf($json->services) ){
                        
                    }else{
                        if( property_exists($json,'services') && property_exists( $json->services, $service_name) ){
                            $services[ $service_name ] = new DHL_Service_dhlRetoure([
                                'billingNumber'=>self::get_dhlRetoure_billingNumber($json->product),
                                'returnAddress'=>new DHL_FullAdress([
                                    'name1'=>$json->services->$service_name->returnAddress->name1,
                                    'addressStreet'=>$json->services->$service_name->returnAddress->addressStreet,
                                    'postalCode'=>$json->services->$service_name->returnAddress->postalCode,
                                    'city'=>$json->services->$service_name->returnAddress->city,
                                    'country'=>$json->services->$service_name->returnAddress->country,
                                ])
                            ]);
                        }
                    }
                    break;
            }
        }
        $customs_save = false;
        if(isset($json->customs)){
            $customs_save = true;
            $items = [];
            $customs = new DHL_customs([
                'invoiceNo'=>$json->customs->invoiceNo,
                'exportType'=>$json->customs->exportType,
                'exportDescription'=>(isset($json->customs->exportDescription) and !empty($json->customs->exportDescription))?$json->customs->exportDescription:'',
                // HABEN WIR GAR NICHT IM CONFIG TEMPLATE
                'shippingConditions'=>(isset($json->customs->shippingConditions) and !empty($json->customs->shippingConditions))?$json->customs->shippingConditions:'',
                'hasElectronicExportNotification'=>(isset($json->customs->hasElectronicExportNotification))?$json->customs->hasElectronicExportNotification:false,
                'MRN'=>(isset($json->customs->MRN) and !empty($json->customs->MRN))?$json->customs->MRN:'',
                'postalCharges'=>new DHL_Service_Amount([
                    'currency'=>$json->customs->postalCharges->currency,
                    'value'=>(float)$json->customs->postalCharges->value,
                ]),
                'shipperCustomsRef'=>(isset($json->customs->shipperCustomsRef) and !empty($json->customs->shipperCustomsRef))?$json->customs->shipperCustomsRef:'',
                'consigneeCustomsRef'=>(isset($json->customs->consigneeCustomsRef) and !empty($json->customs->consigneeCustomsRef))?$json->customs->consigneeCustomsRef:'',
                'items'=>$items
            ]);
        }
        if($customs_save){
            $data = [
                'product'=>$json->product,
                'customs'=>$customs,
            ];
	    if(is_countable($services) and sizeOf($services)){
                $data['services']=$services;
            }
            return new DHL_shipment($data);
        }else{
	    $data = [
                'product'=>$json->product,
            ];
	    if(is_countable($services) and sizeOf($services)){
                $data['services']=$services;
            }
            return new DHL_shipment($data);
        }
    }

    public static function fillRefNo($refNo){
        if(strlen($refNo)<8){
            $refNo .= '-';
        }
        while(strlen($refNo)<8){
            $refNo .= '0';
        }
        return $refNo;
    }
    
    public static function get_refNo_from_source_object( $obj ){
        if(get_class($obj) == 'order'){
            $str = $obj->info['id'];
        }else{
            $str = 'R'.$obj->info['id'];
        }
        return self::fillRefNo($str);
    }
    
    /**
     * Validiert den Versand fuer eine order/rekla (Vor Uebergabe in Versand)
     * 
     * @params object $dhlObj
     * @params object $srcObj order oder rekla
     */
    public function validate_or( $dhlObj, $srcObj ){
        $dhlObj->consignee = $this->get_consignee_data_from_source_object($srcObj);
        $dhlObj->shipper = $this->get_shipper();
        $dhlObj->refNo = self::get_refNo_from_source_object($srcObj);
        $dhlObj->shipDate = date('Y-m-d');
        if(isset($_GET['weight']) and floatval($_GET['weight']) > 0){
            $dhl_weight = floatval($_GET['weight'])/1000;
        }else{
            $dhl_weight = floatval(str_replace(',','.',constant('SHIPPING_BOX_WEIGHT')));
            $dhl_weight += $srcObj->show_weight(0); // 0 = noch keine shippings_id
            if($dhl_weight < 1){
                $dhl_weight = 1;
            }
        }
        $dhlObj->details = new DHL_Details([
            /*
            'dim'=>new DHL_Dim([
                'uom'=>'mm',
                'height'=>100,
                'length'=>200,
                'width'=>150
            ]),*/
            'weight'=>new DHL_Weight([
                'uom'=>'g',
                'value'=>(int)($dhl_weight*1000)
            ])
        ]);
        $dhlObj->billingNumber = self::get_billingNumber($dhlObj->product);
        $dhlObj->product = substr($dhlObj->product,0,-4);
        return $this->createShipmentOrder([$dhlObj],true);
    }
    
    public static function get_response_errors( object $res ){
        $msg = [];
        if(!is_countable($res->items)){
            if(isset($res->status) and gettype($res->status) == 'integer' and $res->status != 200){
                $msg[] = sprintf(
                    '[Error #%d] %s: %s',
                    $res->status,
                    $res->title,
                    $res->detail
                );
            }
            return $msg;
        }
        if($res->status->statusCode != 200){
            foreach($res->items as $it){
                $vms = [];
                foreach($it->validationMessages as $vm){
                    $vms[] = sprintf('[%s:%s] %s',
                        $vm->validationState,$vm->property,$vm->validationMessage
                    );
                }
                $msg[] = sprintf('%s<br />%s',$it->sstatus->title,
                        implode('<br />',$vms)
                );
            }
        }
        return $msg;
    }
    
    public static function get_country_based_shipper_profile( string $country_name, int $countries_id ){
        $int = false;
        $ger = false;
        $eur = false;
        if($country_name == 'Germany'){
            $ger = true;
        }else{
            $eu_countries_query = xtc_db_query(sprintf(
                "SELECT countries_group FROM countries_allocations WHERE countries_id='%d'",
                $countries_id
            ));
            if(!xtc_db_num_rows($eu_countries_query)){
                throw new Exception(sprintf(
                    "Das Land %s in der Versandadresse konnte weder EU noch Weltweit zugeordnet werden.",
                    $country_name
                ));
            }

            while($record = xtc_db_fetch_array($eu_countries_query)){
                switch($record['countries_group']){
                    case 'EuropeanUnion':
                        $eur = true;
                        break;
                    default:
                        $int = true;
                        break;
                }
            }
        }
        $sp = new \YES4Trade\Model\shipper_profiles;
        $PRODUKTE = self::PRODUKTE;
        $V54_exists = false;
        foreach($sp->get_list() as $p){
            if($p['versender'] == 'DHL'){
                if(empty($p['versender_params'])){
                    continue;
                }
                $json = json_decode($p['versender_params']);
                if($json->product == 'EU'){
                    $V54_exists = true;
                }
            }
            
        }
        foreach($sp->get_list() as $p){
            if($p['versender'] == 'DHL'){
                    $params = json_decode($p['versender_params']);
                    $current_pr = false;
                    foreach($PRODUKTE as $pr){
                        if($pr['id'] == $params->product){
                            $current_pr = $pr;
                        }
                    }
                    if(stristr($current_pr['GKV_API'],'V01') === false and stristr($current_pr['GKV_API'],'V62') === false){
                        if($eur === true && $V54_exists){
                            if(substr($current_pr['GKV_API'],0,3) == 'V54'){
                                    $default_value = $p['profile_name'];
                                    break;
                            }
                        }else{
                            // international
                            if($int === true or !$V54_exists){
                                    $default_value = $p['profile_name'];
                                    break;
                            }
                        }
                    }else{
                            // national
                            if($ger === true){
                                    $default_value = $p['profile_name'];
                                    break;
                            }
                    }
            }
        }
        return $default_value;
    }
}

/**
 * API KEY
Der Schlüssel ist ein hexadezimaler String wie unten angeführt.
Erstellen Sie einen http-Header wie im curl-Beispiel zur Validierung:
curl -H "accept: application/json" -H "dhl-api-key: ${KEY}" https://api-sandbox.dhl.com/parcel/de/shipping/v2/"
Beispiel Schlüssel: 7eelE1paJtbWvAK
 
 * GKP Benutzername und Passwort 
muss via basic authentication bereitgestellt werden. Hier ist ein Curl Beispiel.
curl -u ${gkpuser}:${gkppass} -H "dhl-api-key: ${KEY}" https://api-sandbox.dhl.com/parcel/de/shipping/v2/
 * 
 */
    
